#include <zephyr.h>
#include <device.h>
#include <devicetree.h>
#include <drivers/gpio.h>
#include <drivers/spi.h>
#include <sys/printk.h>

#include "ST7789.h"

#define LCD_BACKLIGHT 6
#define LCD_RST 25
#define LCD_DC 26
#define LCD_TE 10

const struct device *spi_dev;
const struct device *lcd_dev;

uint8_t window[4] = {0, 0, 0, 240};

struct k_poll_signal lcd_spi_signal;
struct k_poll_event lcd_spi_event = K_POLL_EVENT_INITIALIZER(K_POLL_TYPE_SIGNAL, K_POLL_MODE_NOTIFY_ONLY, &lcd_spi_signal);

struct spi_cs_control spi_cs = {
    .gpio_dev = NULL,
    .gpio_pin = 7,
    .gpio_dt_flags = GPIO_ACTIVE_LOW,
    .delay = 0,
};

static const struct spi_config spi_cfg = {
    .operation = SPI_WORD_SET(8) | SPI_TRANSFER_MSB |
                 SPI_MODE_CPOL | SPI_MODE_CPHA,
    .frequency = 20000000,
    .slave = 0,
    .cs = &spi_cs};

static void spi_init(void)
{
    const char *const spiName = "SPI_4";
    spi_dev = device_get_binding(spiName);

    if (spi_dev == NULL)
    {
        printk("Could not get %s device\n", spiName);
        return;
    }
    else
        printk("found spi device!");
}

void spi_send_byte(uint8_t data)
{
    static uint8_t tx_buffer[1];
    tx_buffer[0] = data;

    const struct spi_buf tx_buf = {
        .buf = tx_buffer,
        .len = sizeof(tx_buffer)};

    const struct spi_buf_set tx = {
        .buffers = &tx_buf,
        .count = 1};

    spi_write(spi_dev, &spi_cfg, &tx);
}

void spi_send_bytes(uint8_t *data, int length)
{
    const struct spi_buf tx_buf = {
        .buf = data,
        .len = length};

    const struct spi_buf_set tx = {
        .buffers = &tx_buf,
        .count = 1};

    spi_write(spi_dev, &spi_cfg, &tx);
}

void spi_send_bytes_async(uint8_t *data, int length)
{
    const struct spi_buf tx_buf = {
        .buf = data,
        .len = length};

    const struct spi_buf_set tx = {
        .buffers = &tx_buf,
        .count = 1};

    spi_write_async(spi_dev, &spi_cfg, &tx, &lcd_spi_signal);
}

void lcd_send_command(uint8_t command)
{
    gpio_pin_set(lcd_dev, LCD_DC, 0);

    spi_send_byte(command);
}

void lcd_send_bytes(uint8_t *data, int length)
{
    gpio_pin_set(lcd_dev, LCD_DC, 1);

    spi_send_bytes(data, length);
}

void lcd_send_bytes_async(uint8_t *data, int length)
{
    gpio_pin_set(lcd_dev, LCD_DC, 1);

    spi_send_bytes_async(data, length);
}

void lcd_send_byte(uint8_t data)
{
    gpio_pin_set(lcd_dev, LCD_DC, 1);

    spi_send_byte(data);
}

void lcd_off()
{
    gpio_pin_set(lcd_dev, LCD_BACKLIGHT, 0);
}

void lcd_on()
{
    gpio_pin_set(lcd_dev, LCD_BACKLIGHT, 1);
}

void lcd_backlight_off()
{
    gpio_pin_set(lcd_dev, LCD_BACKLIGHT, 1);
}
void lcd_backlight_on()
{
    gpio_pin_set(lcd_dev, LCD_BACKLIGHT, 1);
}

void lcd_configure()
{
    k_poll_signal_init(&lcd_spi_signal);
    lcd_spi_signal.signaled = 0;

    lcd_dev = device_get_binding("GPIO_0");
    spi_cs.gpio_dev = lcd_dev;

    gpio_pin_configure(lcd_dev, LCD_BACKLIGHT, GPIO_OUTPUT_ACTIVE);
    gpio_pin_configure(lcd_dev, LCD_DC, GPIO_OUTPUT_ACTIVE | GPIO_PULL_UP);
    gpio_pin_configure(lcd_dev, LCD_RST, GPIO_OUTPUT_ACTIVE | GPIO_PULL_UP);
    gpio_pin_configure(lcd_dev, LCD_TE, GPIO_INPUT);

    gpio_pin_set(lcd_dev, LCD_RST, 1);
    k_msleep(150);
    gpio_pin_set(lcd_dev, LCD_RST, 0);
    k_msleep(150);
    gpio_pin_set(lcd_dev, LCD_RST, 1);
    k_msleep(150);

    lcd_on();

    spi_init();

    lcd_send_command(ST7789_SWRESET); //reset
    k_msleep(6);
    lcd_send_command(ST7789_SLPOUT); //wake
    k_msleep(120);
    lcd_send_command(ST7789_COLMOD); //set color mode
    lcd_send_byte(0x55);             //01010101 16 bit color mode
    k_msleep(6);
    lcd_send_command(ST7789_MADCTL); 
    lcd_send_byte(0x0);
    lcd_send_command(ST7789_CASET); //set the colum (x) min and max (0,240)
    lcd_send_bytes(window, 4);
    lcd_send_command(ST7789_RASET); //set the row (y) min and max (0x240)
    lcd_send_bytes(window, 4);
    lcd_send_command(ST7789_INVON); //invert the display
    k_msleep(10);
    lcd_send_command(ST7789_NORON); //set display mode to normal (vice partial)
    k_msleep(10);
    lcd_send_command(ST7789_TEON);   //turn the tear flag on
    lcd_send_byte(0x0); //vlbank only
    lcd_send_command(ST7789_DISPON); //turn the display on
    k_msleep(255);

}

uint16_t front_buffer[240 * 240];

uint16_t *lcd_get_buffer() { return front_buffer; }

void lcd_update_screen()
{
    static int first = 1;

    if(!first)
    {
        k_poll(&lcd_spi_event, 1, K_FOREVER);
        lcd_spi_signal.signaled = 0;
        lcd_spi_event.state = K_POLL_STATE_NOT_READY;
    }
   
    first = 0;

    lcd_send_command(ST7789_CASET); 
    lcd_send_bytes(window, 4);
    lcd_send_command(ST7789_RASET); 
    lcd_send_bytes(window, 4);

    lcd_send_command(ST7789_RAMWR);

    while (gpio_pin_get_raw(lcd_dev, LCD_TE));

    lcd_send_bytes_async((uint8_t *)front_buffer, 240 * 240*2);
}